/* ###########################################################################
#
# Copyright (c) 1999-2002 Atmel Corporation  All Rights Reserved
#
# Atmel Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Atmel, Incorporated ("Atmel").  It may only be read,
# used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Atmel, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Atmel.  If there
# is any doubt as to whether a party is entitled to access, read, use, copy,
# adapt, modify or otherwise utilize the Software, or whether a party is
# entitled to disclose the Software to any other party, you should contact
# Atmel.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Atmel granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Atmel.
#
# ##########################################################################*/


#include "fs_hal.h"
#include "fs_flash.h"
#include "fs_driver.h"

#ifdef WEAR_LEVEL_ENABLED


/* --------------------------------------------------------------------------
 *
 *  Update_Erase_Count
 *
 *      This function will update the page erase count.
 *      This count is stored in the first three bytes of
 *      each page. This function will read directly from the
 *      flash increament the count and store the count back
 *      into the buffer. It is assumed that a program will
 *      happen from another function to store the count back.
 *
 *  Parameters:
 *      uint16 page_number
 *
 *  Returns:
 *      uint32 count
 *
 *
 *  Notes:
 *      None.
 *
 * --------------------------------------------------------------------------
 */
uint32 Update_Erase_Count(uint16 page_number)
{
    uint8   eraseCount[3];
    uint32  count = 0;

    Read_Flash( page_number,0, &eraseCount[0],3);
    eraseCount[0]++;
    if (eraseCount[0] == 0)
    {
        eraseCount[1]++;
        if (eraseCount[1] == 0)
           eraseCount[2]++;
    }
    count = eraseCount[0];
    count &= 0x00FF;
    count += (eraseCount[1]<<8);
    count &= 0x0000FFFF;
    count += (eraseCount[2]<<16);
    count &= 0x00FFFFFF;


    Write_Buffer(0, &eraseCount[0], 3);

    return count;
}
/* --------------------------------------------------------------------------
 *
 *  Initialize_Erase_Count
 *
 *      Initializes the Erase count when wear leveling is enabled.
 *      Each page has a 3 byte field that keeps track of the erase
 *      count for that page. This function starts them off at zero
 *
 *  Parameters:
 *      None
 *
 *  Returns:
 *      None
 *
 *
 *  Notes:
 *      None.
 *
 * --------------------------------------------------------------------------
 */
void Initialize_Erase_Count(void)
{
    uint8   eraseCount[3];
    uint16  start,page;
   uint8 address[4];
    eraseCount[0] = 0;
    eraseCount[1] = 0;
    eraseCount[2] = 0;

    start = FLASH_FILE.file_system_start+1;
    for (page = start;page<(FLASH_FILE.file_system_size+start);page++)
    {
        Write_Buffer(0, &eraseCount[0], 3);
        setupFlashAddress(&address[0],page, 0,FLASH_FILE.buffer_size);
        sendFlashCommand(BUFFER1_PROGRAM_ERASE,&address[0]);
        flashBusyDelay();     /* Delay until the page is programmed */
    }
}

void UpdateWearCount(void)
{
    FLASH_STORAGE.wearCount++;
    return;
}

#endif

