

#include "fs_hal.h"
#include "fs_flash.h"
#define file_system_ram_c
    #include "fs_driver.h"
#undef file_system_ram_c


/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            sendFlashCommand
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               02/13/2002
//;
//;     Inputs:             uint16 command
//;                         uint8 * address_pointer
//;
//;     Outputs:            TRUE or status from flash.
//;
//;     Description:        This function is a generic function that setsup
//;                         the correct command to the Atmel flash part.
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
uint16 sendFlashCommand(uint16 command,uint8 * address_pointer)
{
    uint16 status = 1;
    Select_Flash();
    if (command == STATUS_COMMAND)
    {
        Send_Byte(command);
        status = Get_Byte();
	    Deselect_Flash();
	}
	else
	{
        *address_pointer = command;
        Write_Control_Page(address_pointer,4);
        switch    (command)
        {
            case (READ_CONTINUOUS_COMMAND): /* 0x68 or 0xE8 */
                Write_Control_Page(address_pointer,4);/* SEND 4 DUMMY BYTES */
                break;
            case (READ_MAIN_MEMORY_COMMAND): /* 0x52 or 0xE2 */
                Write_Control_Page(address_pointer,4); /* SEND 4 DUMMY BYTES */
                break;
            case (BUFFER1_READ_COMMAND):/*0x54 or 0xD4 */
                Send_Byte(0); /* SEND ONE DUMMY BYTE */
                break;
             #if 0
            //case (BUFFER2_READ_COMMAND):/*0x56 or 0xD6 */
            //    Send_Byte(0); /* SEND ONE DUMMY BYTE */
            //    break;
            #endif



            /* case (BLOCK_ERASE)://0x50  */
            /* case (BUFFER2_WRITE_AND_PROGRAM)://0x85 */
            /* case (COMPARE_MEMORY_TO_BUFFER2)://0x61 */
            /* case (BUFFER1_WRITE)://0x84 DONT WANT TO DESELECT */
            /* case (BUFFER2_WRITE)://0x87 DONT WANT TO DESELECT */
            /* case (BUFFER1_WRITE_AND_PROGRAM): 0x82 DONT WANT TO DESELECT */
               /* break;*/

            case (BUFFER1_PROGRAM_ERASE):/* 0x83 */
            /*case (BUFFER2_PROGRAM_ERASE)://0x86
            case (BUFFER1_PROGRAM)://0x88
            case (BUFFER2_PROGRAM)://0x89
            case (MEMORY_TO_BUFFER2)://0x55

            /* case (BUFFER2_REWRITE)://0x59 */
            case (COMPARE_MEMORY_TO_BUFFER1)://0x60 */
            case (BUFFER1_REWRITE): /*0x58 */
            case (PAGE_ERASE):/* 0x81 */
            case (MEMORY_TO_BUFFER1):/* 0x53 */

            	    Deselect_Flash();
                break;

           default:
                break;
		}
    }
    return status;
}

/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            Move_Flash_To_Buffer
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               02/13/2002
//;
//;     Inputs:             uint16 page_number
//;
//;     Outputs:            none.
//;
//;     Description:        This function will copy the specified page number
//;                         into the ram buffer on the Atmel part.
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */


void Move_Flash_To_Buffer(uint16 page_number)
{
    uint8 address[4];
    uint16 status;
    setupFlashAddress(&address[0],page_number, 0,FLASH_FILE.buffer_size);
    sendFlashCommand(MEMORY_TO_BUFFER1,&address[0]);//MOVE MEMORY TO BUFFER

    /* IF NOT POLLING THEN THE flashBusyDelay WILL TASK SLEEP
    *  THIS COULD TAKE 10 -  20 MSEC. POLLING WILL ONLY TAKE 200 MICRO SECONDS
    *  THE POLLING CODE IS HERE IN CASE THE USER WANTS TO PERFORM A TASK SLEEP
    *  FOR ALL flashBusyDelay COMMANDS BUT DOES NOT WANT TO WAST THE EXTRA TIME
    *  JUST FOR THIS MOVE COMMAND.
    */
#ifndef POLLING
        do {
            status = sendFlashCommand(STATUS_COMMAND,0);
        } while((status&FLASH_BUSY) == 0);
#else
        flashBusyDelay();    /* DELAY UNTIL FINISHED */
#endif
    return;
}

/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            Read_Buffer, Read_Flash
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               02/13/2002
//;
//;     Inputs:             uint16 page_number, byte_location, length
//;                         uint8 * data_pointer
//;
//;     Outputs:            none.
//;
//;     Description:        These functions will read from directly from the
//;                         flash part or from the internal flash ram buffer.
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
void Read_Buffer(uint16 byte_location,uint8 * data_pointer,uint16 length)
{
    uint8 address[4];
    setupFlashAddress(&address[0],0, byte_location,FLASH_FILE.buffer_size);
    sendFlashCommand(BUFFER1_READ_COMMAND,&address[0]); /*READ THE BUFFER */
    Read_Data_Page(data_pointer,length);
    return;
}


void Read_Flash(uint16 page_number,uint16 byte_location, uint8 * data_pointer,uint16 length)
{
    uint8 address[4];
    setupFlashAddress(&address[0],page_number, byte_location,FLASH_FILE.buffer_size);
    sendFlashCommand(READ_CONTINUOUS_COMMAND,&address[0]); /* READ THE FLASH DIRECTLY */
    Read_Data_Page(data_pointer,length);
    return;
}

/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            Refresh_Page
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               07/15/2002
//;
//;     Inputs:             uint16 page_number
//;
//;     Outputs:            none.
//;
//;     Description:        These functions refresh the specified page number
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
void Refresh_Page(uint16 page_number)
{
    uint8 address[4];
    setupFlashAddress(&address[0],page_number, 0,FLASH_FILE.buffer_size);
    sendFlashCommand(BUFFER1_REWRITE,&address[0]);
    flashBusyDelay();     /* DELAY UNTIL THE PAGE IS REFRESHED */
    return;
}



/* **************************************************************************** */



void Write_Buffer(uint16 byte_location, uint8 * data_pointer, uint16 data_size)
{
    uint8 address[4];
    setupFlashAddress(&address[0],0, byte_location,FLASH_FILE.buffer_size);
    sendFlashCommand(BUFFER1_WRITE,&address[0]);
    Write_Data_Page(data_pointer,data_size);
    return;
}
void Erase_Page(uint16 page_number)
{
    uint8 address[4];
    uint32 count;
    PAGE_ERASE_COUNT(count,page_number)
    setupFlashAddress(&address[0],page_number, 0,FLASH_FILE.buffer_size);
    sendFlashCommand(PAGE_ERASE,&address[0]);
    flashBusyDelay();     /* DELAY UNTIL THE PAGE IS ERASED */
    SECTOR_ERASE_COUNT(page_number);
    return;
}

void Program_Buffer(uint16 page_number)
{
    uint8 address[4];
    uint32 count;
    PAGE_ERASE_COUNT(count,page_number)
    setupFlashAddress(&address[0],page_number, 0,FLASH_FILE.buffer_size);
    sendFlashCommand(BUFFER1_PROGRAM_ERASE,&address[0]);
    flashBusyDelay();     /* DELAY UNTIL THE PAGE IS PROGRAMMED */
    SECTOR_ERASE_COUNT(page_number);
    return;
}

uint32 Write_Flash_Data(uint16 page_number, uint16 byte_location, uint8 * data_pointer, uint16 data_size)
{
    uint8 address[4];
    uint32 count;
    Move_Flash_To_Buffer(page_number);
    PAGE_ERASE_COUNT(count,page_number)
    setupFlashAddress(&address[0],page_number, byte_location,FLASH_FILE.buffer_size);
    sendFlashCommand(BUFFER1_WRITE_AND_PROGRAM,&address[0]);
    Write_Data_Page(data_pointer,data_size);
    flashBusyDelay();     /* DELAY UNTIL THE PAGE IS PROGRAMMED */
    SECTOR_ERASE_COUNT(page_number);
    return count;
}

uint16 FlashCompare(uint16 page_number)
{
    uint8 address[4];
    uint16 status;
    setupFlashAddress(&address[0],page_number, 0,FLASH_FILE.buffer_size);
    sendFlashCommand(COMPARE_MEMORY_TO_BUFFER1,&address[0]);
#ifndef POLLING
        do {
            status = sendFlashCommand(STATUS_COMMAND,0);
        } while((status&FLASH_BUSY) == 0);
#else
        status = flashBusyDelay();    /* DELAY UNTIL FINISHED */
#endif


    if ( (status&FLASH_COMP) == 0) /* if FLASH_COMP is not set no error */
    {
        return DEVICE_SUCCESS;
    }
    return DEVICE_FAIL;

}

/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            setupFlashAddress
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               03/20/2002
//;
//;     Inputs:             none
//;
//;     Outputs:            none.
//;
//;     Description:        This function converts the page number and the
//;                         address within the page into data the flash part
//;                         can utilize. The page number is the page number
//;                         to use to calculate the address. The page address
//;                         is the address inside the page starting at zero.
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
void setupFlashAddress(uint8 * address, uint16 page_number, uint16 page_address,enum FLASH_BUFFER_SIZE size)
{
    *(address + 3) = (uint8)page_address;
    *(address + 2) = 0;
    *(address + 2) = (uint8)(page_address>>8);

    switch (size)
    {
        case (BUFFER_264):
            page_number = page_number*2;
            *(address + 2) = (*(address + 2)|((uint8)(page_number<<1)));
            *(address + 1) = (uint8)(page_number>>7);
            break;
        case (BUFFER_528):
            *(address + 2) = (*(address + 2)|((uint8)(page_number<<2)));
            *(address + 1) = (uint8)(page_number>>6);
            break;
        case (BUFFER_1056):
            *(address + 2) = (*(address + 2)|((uint8)(page_number<<3)));
            *(address + 1) = (uint8)(page_number>>5);
            break;
        case (BUFFER_2112):
            *(address + 2) = (*(address + 2)|((uint8)(page_number<<4)));
            *(address + 1) = (uint8)(page_number>>4);
            break;
    }
    return;
}
/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            flashBusyDelay
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               02/13/2002
//;
//;     Inputs:             none
//;
//;     Outputs:            none.
//;
//;     Description:        This function will delay while the flash is busy.
//;                         If the user has not enabled POLLING then the #define
//;                         FLASH_TASK_SLEEP should be defined to an os sleep
//;                         function for 1 os tick. This will allow the users system
//;                         to continue running while the flash is busy.
//;                         The POLLING and FLASH_TASK_SLEEP defines are set
//;                         in the file "fs_flash.h"
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
uint16 flashBusyDelay(void)
{
        uint16 status;
#ifdef POLLING
        do {
            status = sendFlashCommand(STATUS_COMMAND,0);
        } while((status&FLASH_BUSY) == 0);
#else
        do {
            FLASH_TASK_SLEEP
            status = sendFlashCommand(STATUS_COMMAND,0);
        } while((status&FLASH_BUSY) == 0);

#endif
    return status;
}
/*
//;*************************************************************************
//;_________________________________________________________________________
//; Copyright (c) 1999 - 2002 Atmel Corporation  All Rights Reserved
//;_________________________________________________________________________
//;
//;     Filename            Query_Flash_Status
//;
//;     Author:             DSP OS Inc.
//;
//;     Date:               02/13/2002
//;
//;     Inputs:             none
//;
//;     Outputs:            none.
//;
//;     Description:        This function will read the flash and determine
//;                         which part is being communicated with. If Refresh
//;                         is enabled then the sector erase count array will
//;                         be cleared if the flash copy is not initialized.
//;
//;     Cautions:           None
//;_________________________________________________________________________
//;     Revision History:
//;
//;     Version             Date    Modified By     Change Discription
//;     _______             ____    ___________     __________________
//;_________________________________________________________________________
//;************************************************************************* */
void Query_Flash_Status(void)
{
    uint16 status,i;
    FLASH_FILE.file_system_start = FIRST_FILE_PAGE;

    status = sendFlashCommand(STATUS_COMMAND,0);
    status = status&0x3C; /* FLASH SIZE MASK */
    status = status >>2;
    FLASH_FILE.buffer  = 264;
    FLASH_FILE.buffer_overhead  = 8;
    FLASH_FILE.buffer_size = BUFFER_264;
    switch (status)
    {
        case SIZE_1_MEG:
            FLASH_FILE.flash_size = SIZE_1_MEG;
            FLASH_FILE.file_system_size = 511 - FIRST_FILE_PAGE ;
            break;
        case SIZE_2_MEG:
            FLASH_FILE.flash_size = SIZE_2_MEG;
            FLASH_FILE.file_system_size = 1023 - FIRST_FILE_PAGE ;
            break;
        case SIZE_4_MEG:
            FLASH_FILE.flash_size = SIZE_4_MEG;
            FLASH_FILE.file_system_size = 2047 - FIRST_FILE_PAGE ;
            break;
        case SIZE_8_MEG:
            FLASH_FILE.flash_size = SIZE_8_MEG;
            FLASH_FILE.file_system_size = 4095 - FIRST_FILE_PAGE ;
            break;
        case SIZE_16_MEG:
            FLASH_FILE.buffer  = 528;
            FLASH_FILE.buffer_overhead  = 16;
            FLASH_FILE.buffer_size = BUFFER_528;
            FLASH_FILE.flash_size = SIZE_16_MEG;
            FLASH_FILE.file_system_size = 4095 - FIRST_FILE_PAGE ;
            break;
        case SIZE_32_MEG:
            FLASH_FILE.buffer  = 528;
            FLASH_FILE.buffer_overhead  = 16;
            FLASH_FILE.buffer_size = BUFFER_528;
            FLASH_FILE.flash_size = SIZE_32_MEG;
            FLASH_FILE.file_system_size = 8191 - FIRST_FILE_PAGE ;
            break;
        case SIZE_64_MEG:
            FLASH_FILE.buffer  = 1056;
            FLASH_FILE.buffer_overhead  = 32;
            FLASH_FILE.buffer_size = BUFFER_1056;
            FLASH_FILE.flash_size = SIZE_64_MEG;
            FLASH_FILE.file_system_size = 8191 - FIRST_FILE_PAGE ;
            break;
        case SIZE_128_MEG:
            FLASH_FILE.buffer  = 1056;
            FLASH_FILE.buffer_overhead  = 32;
            FLASH_FILE.buffer_size = BUFFER_1056;
            FLASH_FILE.flash_size = SIZE_128_MEG;
            FLASH_FILE.file_system_size = 16383 - FIRST_FILE_PAGE ;
            break;
        case SIZE_256_MEG:
            FLASH_FILE.buffer  = 2112;
            FLASH_FILE.buffer_overhead  = 64;
            FLASH_FILE.buffer_size = BUFFER_2112;
            FLASH_FILE.flash_size = SIZE_256_MEG;
            FLASH_FILE.file_system_size = 16383 - FIRST_FILE_PAGE ;
            break;
    }
    FLASH_FILE.percent_size = (FLASH_FILE.file_system_size/100);
    if (FLASH_FILE.buffer < 512)
    {
        /* IF START PAGE IS EVEN NEED TO ADD ONE DUE TO 512 BYTE BLOCKS */
        /* NEED TO START ON AN EVEN BOUNDRY FOR PAGES THAT ARE 264 */
        if((FIRST_FILE_PAGE&1) == 0)
        {
            FLASH_FILE.file_system_start = FIRST_FILE_PAGE+1;
        }
    }
#if  (defined(REFRESH_ENABLED)||defined(WEAR_LEVEL_ENABLED))


    Read_Flash(FLASH_FILE.file_system_start,16, (uint8 *)&FLASH_STORAGE,sizeof( FLASH_STORAGE));
    if ( FLASH_STORAGE.valid_data != 0x55AA)
    {
        /* INITIALIZE THE ERASE ARRAY TO ZERO */
        for (i = 0; i <= SECTOR_ARRAY_SIZE; i++)
        {
            FLASH_STORAGE.sector_array[i] = 0;
        }
        FLASH_STORAGE.valid_data = 0x55AA;
        FLASH_STORAGE.wearCount = 0;
        Write_Flash_Data(FLASH_FILE.file_system_start,16, (uint8 *)&FLASH_STORAGE,sizeof( FLASH_STORAGE));
        Initialize_Erase_Count();
    }
#endif
    return;
}

