

/* 
 *  Sixpack.h
 *
 *  Header File for the LZSS algorithm (implemented using a Binary Tree).
 *
 *
 *  Description:
 *
 *      See Sixpack.h for a description of the Sixpack Compression utility.
 *
 *
 *      Besides, some slight "C" changes, I added code to place a header on
 *      the file, which contains the size, date, and some other useful info
 *      THAT must not be compressed in final image.
 */

#define __COMPRESS_SOURCE
    #include "Sixpack.h"
#undef __COMPRESS_SOURCE

/*
 *  Globals and such
 */

#if defined(_COMPRESS_BSS_INITIALIZE)

    extern int copymin[];
    extern int copymax[];

    // Huffman tree

    extern short left[];
    extern short right[];
    extern short up[];
    extern short freq[];

#else

    extern int* copymin;
    extern int* copymax;

    extern short* left;
    extern short* right;
    extern short* up;
    extern short* freq;

#endif

void AddNode( int n );
void DeleteNode( int n );
void UpdateModel( int code );
void InitializeHuffman( void );

static long bytes_in;
static long bytes_out;

/* --------------------------------------------------------------------------
 *
 *  Support Methods 
 *
 * --------------------------------------------------------------------------
 */

DEFINE_COMPRESSION_FUNCTION_SUPPORT

DECLARE_COMPRESSION_FUNCTION_SUPPORT

//-----------------------------------------------------------------------------
//  Function: 
//      flush_bits
//
//  Description: 
//      ???
//
//  Cautions:
//      Converted to a macro to improve speed somewhat...
//
//  Functions Called:
//      putc
//
//  Called By:
//
//-----------------------------------------------------------------------------

#define flush_bits(output) \
    if (output_bit_count > 0) \
    { \
        PUTCHAR(pOutput,(output_bit_buffer << (8-output_bit_count))); \
        bytes_out++; \
    }

//-----------------------------------------------------------------------------
//  Function: 
//      input_bit
//
//  Description: 
//      Reads an bit from input file
//
//  Cautions:
//      None
//
//  Functions Called:
//      None
//
//  Called By:
//
//-----------------------------------------------------------------------------

int input_bit( compress_data_t* pInput )
{
    int bit;

    if (input_bit_count-- == 0) 
    {
        input_bit_buffer = GETCHAR(pInput);
        input_bit_count = 7;
    }
  
    bit = (input_bit_buffer & 0x80) != 0;
    input_bit_buffer <<= 1;

    return bit;
}

//-----------------------------------------------------------------------------
//  Function: 
//      output_bit
//
//  Description: 
//      Writes one bit to output file
//
//  Cautions:
//      None
//
//  Functions Called:
//      putc
//
//  Called By:
//
//-----------------------------------------------------------------------------

static void output_bit( compress_data_t* pOutput, int bit)
{
    output_bit_buffer <<= 1;

    if ( bit ) 
        output_bit_buffer |= 1;
  
    if ( ++output_bit_count == 8 ) 
    {
        PUTCHAR(pOutput,output_bit_buffer);
        output_bit_count = 0;
        ++bytes_out;
    }
}

//-----------------------------------------------------------------------------
//  Function: 
//      input_code
//
//  Description: 
//      Reads Multibit code from Input File
//
//  Cautions:
//      None
//
//  Functions Called:
//      input_bit
//
//  Called By:
//
//-----------------------------------------------------------------------------

int input_code( compress_data_t* pInput, int bits )
{
    int i, bit = 1, code = 0;

    for ( i = 0; i < bits; i++ ) 
    {
        if ( input_bit(pInput) ) 
            code |= bit;

        bit <<= 1;
    }

    return code;
}

//-----------------------------------------------------------------------------
//  Function: 
//      output_code
//
//  Description: 
//      Writes multibit code to output file
//
//  Cautions:
//      None
//
//  Functions Called:
//      output_bit      
//
//  Called By:
//
//-----------------------------------------------------------------------------

static void output_code( compress_data_t* pOutput, int code, int bits )
{
    int i;

    for ( i = 0; i < bits; i++ ) 
    {
        output_bit( pOutput, code & 0x0001 );
        code >>= 1;
    }
}

//-----------------------------------------------------------------------------
//  Function: 
//      char_uncompress( unsigned char* )
//
//  Description: 
//      Uncompress a character code from input stream
//
//  Cautions:
//      None
//
//  Functions Called:
//      input_bit, UpdateModel
//
//  Called By:
//
//-----------------------------------------------------------------------------

int char_uncompress( compress_data_t* pInput )
{
    int a = ROOT;

    do 
    {
        if ( input_bit(pInput) ) 
            a = right[a];
        else 
            a = left[a];
    } 
    while ( a <= MAXCHAR );
  
    a -= SUCCMAX;

    UpdateModel( a );
  
    return a;
}

//-----------------------------------------------------------------------------
//  Function: 
//      compress
//
//  Description: 
//      Compress a character code to output stream
//
//  Cautions:
//      None
//
//  Functions Called:
//      output_bit, UpdateModel
//
//  Called By:
//
//-----------------------------------------------------------------------------

static void char_compress( compress_data_t* pOutput, int code )
{
    int a, sp = 0;
    int stack[50];

    a = code + SUCCMAX;
    do 
    {
        stack[sp++] = ( right[up[a]] == a );
        a = up[a];
    } 
    while (a != ROOT);

    do 
    {
        output_bit(pOutput,stack[--sp]);
    } 
    while (sp);
  
    UpdateModel(code);
    return;
}

//-----------------------------------------------------------------------------
//  Function: 
//      match
//
//  Description: 
//      Find longest string matching lookahead buffer string 
//
//  Cautions:
//      None
//
//  Functions Called:
//      getkey
//
//  Called By:
//
//-----------------------------------------------------------------------------

int match( int n, int depth )
{
    int i, j, index, key, dist, len, best = 0, count = 0;

    if (n == maxsize) n = 0;
    key = getkey(n);
    index = head[key];

    while (index != NIL) 
    {
        // Did we exceed depth? (Quit)

        if ( ++count > depth ) 
            break;

        if ( buffer[(n+best)%maxsize] == buffer[(index+best)%maxsize] ) 
        {
            len = 0;  
            i = n;  
            j = index;

            while ( (buffer[i] == buffer[j]) && (len < MAXCOPY) && 
                                                        (j != n) && (i != insert) )
            {
                ++len;
                if ( ++i == maxsize ) 
                    i = 0;
                if ( ++j == maxsize ) 
                    j = 0;
            }

            dist = n - index;
            if (dist < 0) 
                dist += maxsize;
            dist -= len;

            // If dict file, quit at shortest distance range

            if ( dictfile && dist > copymax[0] ) 
                break;

            // Update best match

            if ( ( len > best ) && ( dist <= maxdistance ) )
            {     
                if ( ( len > MINCOPY ) || ( dist <= copymax[SHORTRANGE+binary] ) )
                {
                    best = len; 
                    distance = dist;
                }
            }
        }
        index = succ[index];
    }

    return best;
}

//-----------------------------------------------------------------------------
//  Finite Window compression routines

#define IDLE    0      // Not processing a copy
#define COPY    1      // Currently processing copy

//-----------------------------------------------------------------------------
//  Function: 
//      void dictionary( void )
//
//  Description: 
//      Check first buffer for ordered dictionary file. Better compression 
//      using short distance copies.
//
//  Cautions:
//      None      
//
//  Functions Called:
//      
//
//  Called By:
//
//-----------------------------------------------------------------------------

void dictionary( void )
{
    int i = 0, j = 0, k, count = 0;

    // Count matching chars at start of adjacent lines
  
    while (++j < MINCOPY+MAXCOPY) 
    {
        if (buffer[j-1] == 10) 
        {
            k = j;
            while (buffer[i++] == buffer[k++]) 
                ++count;
            i = j;
        }
    }

    // If matching line prefixes > 25% assume dictionary

    if ( count > (MINCOPY+MAXCOPY) / 4 ) 
        dictfile = 1;

    return;
}

//-----------------------------------------------------------------------------
//  Function: 
//      encode( FILE*, FILE* )
//
//  Description: 
//      Encode file from input to output
//
//  Cautions:
//      None
//
//  Functions Called:
//      malloc, fseek, fputc, compress, flush_bits, getc, DeleteNode, 
//      AddNode, match, output_code
//
//  Called By:
//
//-----------------------------------------------------------------------------

compress_size_t Compress(
    compress_data_t* pInput, 
    compress_data_t* pOutput 
)
{
    int c, i, n=MINCOPY, addpos=0, full=0, state=IDLE, nextlen;
    compress_size_t len;

    InitializeHuffman();

    head = malloc((unsigned long)HASHSIZE*sizeof(short));
    tail = malloc((unsigned long)HASHSIZE*sizeof(short));
    succ = malloc((unsigned long)maxsize*sizeof(short));
    pred = malloc((unsigned long)maxsize*sizeof(short));

    buffer = (unsigned char *) malloc(maxsize*sizeof(unsigned char));

    do
    {
        /* 
         * Verify Memory allocation
         */

        if ( 
            ( head==NULL ) || 
            ( tail==NULL ) || 
            ( succ==NULL ) || 
            ( pred==NULL ) || 
            ( buffer==NULL ) ) 
        {
            bytes_out = COMPRESSION_MEMORY;
            break;
        }

        REWIND(pInput);
        REWIND(pOutput);

        /*
         *  Validate Length
         */

        len = GETLEN(pInput);
        if ( len == 0 )
        {
            break;
        }

        /*
         *  The first FOUR bytes are the file's original size (typically, used
         *  for preallocation of buffers).
         *
         *  This is optional and can be removed.
         */

        for( i = 0; i < 4 ; ++i )
        {
            PUTCHAR(pOutput,(unsigned char)( len >> i * 8 ) );
        }

        // Initialize hash table to empty

        for ( i = 0; i < HASHSIZE; i++ ) 
        {
            head[i] = NIL;
        }
        len = 0;

        // OK, now compress first few characters using Pure Huffman

        for ( i = 0; i < MINCOPY; i++ ) 
        {
            if ( (c = GETCHAR(pInput) ) == EOF) 
            {
                char_compress(pOutput,TERMINATE);
                flush_bits(pOutput);

                bytes_out = COMPRESSION_FAIL;
                break;
            }
    
            ++bytes_in;
            char_compress(pOutput,c);
            buffer[i] = c;
        }

        if ( bytes_out == COMPRESSION_FAIL )
        {
            break;
        }

        // Preload next few characters into lookahead buffer (test for binary file)
  
        for (i = 0; i<MAXCOPY; i++) 
        {
            if ( (c = GETCHAR(pInput) ) == EOF) 
            {
                break;
            }

            buffer[insert++] = c;  
            ++bytes_in;
            if ( c > 127 )
            { 
                binary = 1;     /* Binary file ? */
            }
        }
  
        // Check dictionary file

        dictionary();

        while (n != insert) 
        {
    
            // Check compression to insure really a dictionary file

            if ( dictfile && ((bytes_in % MAXCOPY) == 0) )
            {
                if (bytes_in/bytes_out < 2)
                {
                    dictfile = 0;               // Oops, not a dictionary file!
                }
            }

            // Update nodes in hash table lists

            if ( full ) 
            {
                DeleteNode(insert);
            }

            AddNode(addpos);

            // Are we doing copy?
    
            if ( state == COPY ) 
            {
                if ( --len == 1 ) 
                    state = IDLE;
            } 
        
            // Check for new copy 

            else 
            {

                // Get match length at next character and current char

                if ( binary ) 
                {
                    nextlen = match(n+1,BINNEXT);
                    len = match(n,BINSEARCH);
                } 
            
                else 
                {
                    nextlen = match(n+1,TEXTNEXT);
                    len = match(n,TEXTSEARCH);
                }

                // If long enough and no better match at next char, start copy

                if (len >= MINCOPY && len >= nextlen) 
                {
                    state = COPY;

                    // Look up minimum bits to encode distance
        
                    for (i = 0; i<COPYRANGES; i++)
                    {
                        if (distance <= copymax[i]) 
                        {
                            char_compress(
                                pOutput,
                                FIRSTCODE-MINCOPY+len+i*CODESPERRANGE
                            );
                            output_code(
                                pOutput,
                                distance-copymin[i],
                                copybits[i]
                            );
                            break;
                        }
                    }
                }

                // Else output single literal character

                else
                {
                char_compress(pOutput,buffer[n]);
                }
            }

            // Advance buffer pointers

            if (++n == maxsize)
            { 
                n = 0;
            }

            if (++addpos == maxsize) 
            {
                addpos = 0;
            }

            // Add next input character to buffer
    
            if (c != EOF) 
            {
                if ( (c = GETCHAR(pInput)) != EOF)  
                {
                    buffer[insert++] = c;  
                    ++bytes_in;
                } 
        
                else
                {
                    full = 0;
                }
      
                if (insert == maxsize) 
                {
                    insert = 0; 
                    full = 1;
                } 
            }
        }

        // Output EOF code and free memory

        char_compress( pOutput, TERMINATE );
        flush_bits( pOutput );
    }
    while (0);

    /*
     *  Release Memory
     */

    if ( head != NULL ) { free(head); }
    if ( tail != NULL ) { free(tail); }
    if ( succ != NULL ) { free(succ); }
    if ( pred != NULL ) { free(pred); }
    if ( buffer != NULL ) { free(buffer); }

    return bytes_out;
}

//-----------------------------------------------------------------------------
//  Function: 
//      decode( unsigned char*, unsigned char* )
//
//  Description: 
//      Decode file from input to output
//
//  Cautions:
//      None
//
//  Functions Called:
//      malloc, uncompress, input_code, free
//
//  Called By:
//
//-----------------------------------------------------------------------------

compress_size_t Uncompress( 
    compress_data_t* pInput, 
    compress_data_t* pOutput 
)
{
    int c, n;
    int i, j, k;
    int dist, len, index;

    compress_size_t _nExpectLen;

    InitializeHuffman();

    REWIND(pInput);
    REWIND(pOutput);

    n = 0;
    buffer = (unsigned char *)malloc( maxsize );

    // Did we get the buffer???

    if ( !buffer ) 
    {
        return COMPRESSION_MEMORY;
    }

    /*
     *  The first FOUR bytes are the file's original size (typically, used
     *  for preallocation of buffers).
     *
     *  This is optional and can be removed.
     */

    for( i = 0; i < 4 ; ++i )
    {
        _nExpectLen |= ( GETCHAR(pInput) << i * 8 );
    }

    // OK, begin uncompression.

    while ( ( c = char_uncompress(pInput) ) != TERMINATE ) 
    {
         // Single literal character?

        if ( c < 256 ) 
        {
            PUTCHAR(pOutput,c);
            bytes_out++;
            buffer[n++] = c;

            if ( n == maxsize ) 
                n = 0;
        } 

        // Else string copy length/distance codes

        else 
        {
            index = (c - FIRSTCODE)/CODESPERRANGE;
            len = c - FIRSTCODE + MINCOPY - index*CODESPERRANGE;
            dist = input_code( pInput, copybits[index] ) + len + copymin[index];
            j = n; 
            k = n - dist;
            if ( k < 0 ) 
                k += maxsize;
      
            for (i = 0; i < len; i++ ) 
            {
                PUTCHAR(pOutput,buffer[k]);
                bytes_out++;
                buffer[j++] = buffer[k++];

                if ( j == maxsize ) { j = 0; }
                if ( k == maxsize ) { k = 0; }
            }
      
            n += len;
            if ( n >= maxsize ) 
                n -= maxsize;
        }
    }

    free( buffer );

    /*
     *  Did we decompress successfully????
     */

    if ( _nExpectLen != GETPOS(pOutput) )
        return COMPRESSION_FAIL;

    /*
     *  Return the length
     */

    return _nExpectLen;
}

// End of SixPack_Decode.c