

// For debug purposes only

#if defined(WIN32) && defined(_DEBUG)
    #include <stdio.h>
#endif

#include "LZMemory.h"

//-----------------------------------------------------------------------------
//  Memory Management
//
//      Since this is intended to run out of the Bootload we do not have
//      provisions for memory management and thus must do this ourselves.
//      This object acts as a placement holder for memory

LZMemory g_LZMemPool = LZMEMORY_NULL;

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//  Class LZMemory

//-----------------------------------------------------------------------------
// Private Methods

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_IsFree
//
//  Description:
//      Returns if the memory block is free or not.
//
//  Cautions:
//      None
//
//  Functions Called:
//      None
//
//  Called By:
//
//-----------------------------------------------------------------------------

#define LZMemory_IsFree(This) \
    ( (This != LZMEMORY_NULL) && !(This->size & LZMEMORY_STAT) )

//-----------------------------------------------------------------------------
// Constructor/Destructor

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_Construct
//
//  Description:
//      Creates an LZMemory object which is used as a Placement Syntax
//      structure. Total is the total amount of memory allocated for the
//      object minus the object itself which occupies the beginning.
//
//      The memory is setup such that the first block contains an object,
//      LZMemoryBlock, which contains the size (including the block header),
//      and pointer to the next memory block. Initially there is only one
//      big block.
//
//      The highest bit in the size indicates whether the block is free or
//      allocated. If cleared, then the memory block is currently free.
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

LZMemory LZMemory_Construct( void* memory, size_t size )
{
    // We assign the memory object to the first part of the pre-allocated
    // data space above, then the actual memory starts at an offset beyond
    // that.

    LZMemory This = (LZMemory)memory;

    // Size is the total size minus the structure itself.
    //
    //      Remember the upper bit is used to distinguish whether the block
    //      is free or used.

    This->size = ( size - sizeof(struct LZMemory) ) & 0x7FFFFFFFL;
    This->next = LZMEMORY_NULL;

    // Return the object

    return This;
}

//-----------------------------------------------------------------------------
//  Function:
//      LZBuffer_Destroy
//
//  Description:
//
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

void LZMemory_Destroy( LZMemory This )
{
    if ( This )
    {
        This->size = 0;
        This->next = LZMEMORY_NULL;
    }
}

//-----------------------------------------------------------------------------
// Public Methods

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_Allocate
//
//  Description:
//      Allocates memory from the free
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

void* LZMemory_Allocate( LZMemory This, size_t size )
{
    LZMemory block = This;          // Start of memory pool
    void* memory = 0;               // Until we find a valid block

    // Continue until we find a block

    while ( block != LZMEMORY_NULL )
    {
        // Is this block free and big enough?

        if ( LZMemory_IsFree(block) &&  ( size <= block->size ) )
        {
            // Number of bytes leftover in block

            size_t mem_left = block->size - size;

            // DEBUG

            #if defined(WIN32) && defined(_DEBUG)
                printf( "Allocate(%ld)\n",size );
            #endif

            // Is it big enough for a new memory block?
            //
            //      Otherwise we just take the few bytes to the end of this one,
            //      since this is not a smart memory manager.

            if ( mem_left > sizeof(struct LZMemory) )
            {
                LZMemory next_block =
                    (LZMemory)( (LZMEMORY_BYTEPTR)block + sizeof(struct LZMemory) + size );

                next_block->next = block->next;
                next_block->size = mem_left - sizeof(struct LZMemory);

                block->next = next_block;
                block->size = size;
            }

            // Set as allocated and assign to user

            block->size |= LZMEMORY_STAT;
            memory = (LZMEMORY_BYTEPTR)block + sizeof(struct LZMemory);

            // If not, then this part of memory becomes a dead zone (unused and is
            // only recovered during deallocation).

            block = LZMEMORY_NULL;
        }
        else
            block = block->next;
    }

    // Return the block

    return memory;
}

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_Deallocate
//
//  Description:
//      Deallocates the used block and packs freespace...slow but affective.
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

void LZMemory_Deallocate( LZMemory This, void* memory )
{
    LZMemory block = (LZMemory)((LZMEMORY_BYTEPTR)memory - sizeof(struct LZMemory));

    // DEBUG

    #if defined(WIN32) && defined(_DEBUG)
        printf( "Deallocate(%ld)\n",(block->size & ~LZMEMORY_STAT) );
    #endif

    // Mark as a free block

    block->size &= ~LZMEMORY_STAT;

    // Now compress free memory (start at the beginning)?

    block = This;

    // Compress all of free memory

    do
    {
        // If the current block and the next block are free...

        if ( LZMemory_IsFree(block) && LZMemory_IsFree(block->next) )
        {
            block->size += block->next->size + sizeof(struct LZMemory);
            block->next = block->next->next;
        }

        else
            block = block->next;
    }
    while ( block != LZMEMORY_NULL );
}

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_Copy
//
//  Description:
//
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

void* LZMemory_Copy( void *dest, const void *src, size_t count )
{
    size_t i;

    for ( i = 0; i < count; i++ )
        *((LZMEMORY_BYTEPTR)dest + i) = *((LZMEMORY_BYTEPTR)src + i);

    return dest;
}

//-----------------------------------------------------------------------------
//  Function:
//      LZMemory_Copy
//
//  Description:
//
//
//  Cautions:
//
//
//  Functions Called:
//
//
//  Called By:
//
//-----------------------------------------------------------------------------

void* LZMemory_Set( void *dest, int c, size_t count )
{
    size_t i;

    for ( i = 0; i < count; i++ )
        *((LZMEMORY_BYTEPTR)dest + i) = c;

    return dest;
}

// End of LZMemory.c