
/* 
 *  Compress.h
 *
 *  Common Header file for all compression algorithms including basic
 *  data types to abstract Buffered and File source.
 */

#ifndef __COMPRESS_H
#define __COMPRESS_H

/* --------------------------------------------------------------------------
 *      ENVIRONMENT EQUATES
 * --------------------------------------------------------------------------
 */

/* --------------------------------------------------------------------------
 *
 *  FILE SUPPORT -  IF Enabled, the compiler or development platform must 
 *                  HAVE support for Standard I/O, i.e. <stdio.h>. This 
 *                  means that a structure of FILE must be supported as 
 *                  well as the basic Buffered File methods.
 */

#ifndef _COMPRESSION_FILE_SUPPORT

#define _COMPRESSION_FILE_SUPPORT

#endif

/* --------------------------------------------------------------------------
 *
 *  GLOBAL DATA -   IF Enabled, the data uses BSS Uninitialized region of the 
 *                  environment instead of the Heap. While this saves time in
 *                  allocation/deallocation of memory resources, it uses 
 *                  static data space.
 */

#ifndef _COMPRESS_BSS_INITIALIZE

/*  #define _COMPRESS_BSS_INITIALIZE        */

#endif

/* --------------------------------------------------------------------------
 *
 *  STANDARD MEM -  Coincides with Standard I/O in some ways, which requires
 *                  that <memory.h> and <stdlib.h> be supported. If not, 
 *                  then the system must use LZMemory for compression cache.
 *                  Failure to do so will RESULT in link errors.
 */

#ifndef _COMPRESS_USE_STDMEMORY

#define _COMPRESS_USE_STDMEMORY

#endif

/*
 *  Depending on how the user wants to use memory, all arrays can either be
 *  initialized and allocated dynamically or in uninitialized
 */

#if !defined(_COMPRESS_BSS_INITIALIZE)
    #if defined(_COMPRESS_USE_STDMEMORY)
        #include <stdlib.h>
        #include <memory.h>

    /* ----------------------------------------------------------------------
     *  Non-standard memory. LZMemory is included with this project only as a
     *  placeholder. The developer may include their own, but should at least
     *  export the methods for "malloc", "free", "memset", "memcpy" or else
     *  change all this wonderful source code...
     */

    #else
        #include "LZMemory.h"
    #endif
#endif

/* --------------------------------------------------------------------------
 *  Common types within the Compression Library
 *
 *      pData   - Pointer to the Data Buffer
 *      nPos    - Current Read/Write Position in the Buffer
 *      nLen    - Length of the Buffer
 */

typedef long compress_size_t;
struct _compress_string_t
{
    unsigned char* pData;
    compress_size_t nPos;
    compress_size_t nLen;
};

typedef struct _compress_string_t compress_string_t;

/*
 *  COMPRESSION Using FILE Support in the system
 */

#if defined(_COMPRESSION_FILE_SUPPORT)

    #include <stdio.h>

    /*
     *  If FILE Support, then define the types as follows
     */

    typedef enum 
    {
        COMPRESS_FILE_TYPE,
        COMPRESS_STRING_TYPE

    } _DATA_TYPE;

    /*
     *  Compress Object type with Discriminating Union
     *
     *      eType   - Type of Data
     *      pData   - Pointer to where Data is sourced
     */

    struct _compress_data_t
    {
        _DATA_TYPE  eType;
        union
        {
            FILE* file;
            compress_string_t string;
        } pData;
    };

    typedef struct _compress_data_t compress_data_t;

    /*
     *  Defining a Compression Data Type (can be string or file)
     */

    #define DECLARE_COMPRESS_DATA_STRING(nam) \
        compress_data_t nam = { COMPRESS_STRING_TYPE, 0, 0, 0 }

    #define DECLARE_COMPRESS_DATA_FILE(nam) \
        compress_data_t nam = { COMPRESS_FILE_TYPE, (FILE*)0 }

    #define INIT_COMPRESS_DATA_STRING(nam,ptr,len) \
        nam##.pData.string.pData = ptr; \
        if ( ptr != 0 ) nam##->pData.string.nLen = len
        
    #define INIT_COMPRESS_DATA_FILE(nam,ptr) \
        nam##.pData.file = ptr

    #define GET_COMPRESS_DATA_FILE(nam) \
        (FILE*)nam##.pData.file

    #define GET_COMPRESS_DATA_STRING(nam) \
        (compress_string_t*)&(nam##.pData.string)
#else

    /*
     *  Only a string (pointer to a buffer) is valid here
     */

    typedef struct _compress_string_t compress_data_t;

    #define DEFINE_COMPRESS_DATA_STRING( nam, ptr, len ) \
        compress_data_t nam = { ptr, 0, len }

    #ifndef EOF
        #define EOF     (int)-1
    #endif

#endif

/*
 *  Error Codes
 */

#define COMPRESSION_SUCCESS          0
#define COMPRESSION_FAIL            -1
#define COMPRESSION_MEMORY          -2

/*
 *  Application Programming Interface
 *
 *      Each function returns a non-negative number on success
 */

compress_size_t Compress(
    compress_data_t* pInput, 
    compress_data_t* pOutput 
);

compress_size_t Uncompress( 
    compress_data_t* pInput, 
    compress_data_t* pOutput 
);

/*
 *  Application Programming Interface
 */

#if defined(__COMPRESS_SOURCE)

#if defined(_COMPRESSION_FILE_SUPPORT)

    #define DECLARE_COMPRESSION_FUNCTION_SUPPORT \
        static int (*s_fnPutChar[])(compress_data_t*,int) = { \
            PutFileChar, \
            PutStringChar \
        }; \
        static int (*s_fnGetChar[])(compress_data_t*) = { \
            GetFileChar, \
            GetStringChar \
        }; \
        static void (*s_fnRewind[])(compress_data_t*) = { \
            FileRewind, \
            StringRewind \
        }; \
        static compress_size_t (*s_fnGetPos[])(compress_data_t*) = { \
            GetFilePos, \
            GetStringPos \
        }; \
        static compress_size_t (*s_fnGetLen[])(compress_data_t*) = { \
            GetFileLen, \
            GetStringLen \
        }; \

    #define DEFINE_COMPRESSION_FUNCTION_SUPPORT \
        static int PutFileChar(compress_data_t* strm, int data) { \
            return putc(data, strm->pData.file); \
        } \
        static int GetFileChar(compress_data_t* strm) { \
            return getc(strm->pData.file); \
        } \
        static void FileRewind(compress_data_t* strm) { \
            rewind(strm->pData.file); \
        } \
        static compress_size_t GetFilePos(compress_data_t* strm) { \
            return (compress_size_t)ftell(strm->pData.file); \
        } \
        static compress_size_t GetFileLen(compress_data_t* strm) { \
            compress_size_t _nLen; \
            compress_size_t _nCurPos = ftell(strm->pData.file); \
            fseek(strm->pData.file, 0L, SEEK_END); \
	        _nLen = ftell(strm->pData.file); \
            fseek(strm->pData.file, _nCurPos, SEEK_SET); \
	        return _nLen; \
        } \
        static int PutStringChar(compress_data_t* strm, int data) { \
            return ( strm->pData.string.pData[strm->pData.string.nPos++] = data ); \
        } \
        static int GetStringChar(compress_data_t* strm) { \
            if ( strm->pData.string.nPos < strm->pData.string.nLen ) \
                return strm->pData.string.pData[strm->pData.string.nPos++]; \
            else \
                return EOF; \
        } \
        static void StringRewind(compress_data_t* strm) { \
            strm->pData.string.nPos = 0; \
        } \
        static compress_size_t GetStringPos(compress_data_t* strm) { \
            return strm->pData.string.nPos; \
        } \
        static compress_size_t GetStringLen(compress_data_t* strm) { \
            return strm->pData.string.nLen; \
        } \

#else

    #define DECLARE_COMPRESSION_FUNCTION_SUPPORT
    #define DEFINE_COMPRESSION_FUNCTION_SUPPORT

#endif

/*
 *  Application Programming Interface
 */

#if defined(_COMPRESSION_FILE_SUPPORT)

    #define PUTCHAR(strm,data) \
        (*s_fnPutChar[strm->eType])(strm,data)

    #define GETCHAR(strm) \
        (*s_fnGetChar[strm->eType])(strm)

    #define REWIND(strm) \
        (*s_fnRewind[strm->eType])(strm)

    #define GETPOS(strm) \
        (*s_fnGetPos[strm->eType])(strm)
        
    #define GETLEN(strm) \
        (*s_fnGetLen[strm->eType])(strm)
        
#else

    #define PUTCHAR(strm,data) \
        (strm)->pData[(strm)->nPos++] = data

    #define GETCHAR(strm) \
        ( (strm)->nPos >= (strm)->nLen ) ? EOF : (strm)->pData[(strm)->nPos++]

    #define REWIND(strm) \
        (strm)->nPos = 0

    #define GETPOS(strm) \
        (strm)->nPos

    #define GETLEN(strm) \
        (strm)->nLen

#endif

#endif  /*  __COMPRESS_SOURCE */

#endif  /*  __COMPRESS_H */
