

/*
 *  Huffman.c
 */

#define __COMPRESS_SOURCE
    #include "Sixpack.h"
#undef __COMPRESS_SOURCE

//-----------------------------------------------------------------------------
//  Globals

#if defined(_COMPRESS_BSS_INITIALIZE)

    int copymin[COPYRANGES];
    int copymax[COPYRANGES];

#else

    int* copymin;
    int* copymax;

#endif

short copybits[COPYRANGES] =
{
    4, 6, 8, 10, 12, 14              // Distance bits
};

//-----------------------------------------------------------------------------
//

int maxdistance, maxsize;
int distance, insert, dictfile, binary;

//-----------------------------------------------------------------------------
//

short *head, *tail;
short *succ, *pred;
unsigned char *buffer;

//-----------------------------------------------------------------------------
//

#if defined(_COMPRESS_BSS_INITIALIZE)

    short left[MAXCHAR+1];
    short right[MAXCHAR+1];
    short up[TWICEMAX+1];
    short freq[TWICEMAX+1];

#else

    short* left;
    short* right;
    short* up;
    short* freq;

#endif

//-----------------------------------------------------------------------------
//

int     input_bit_count;
int     input_bit_buffer;
int     output_bit_count;
int     output_bit_buffer;
long    bytes_in;
long    bytes_out;

/* --------------------------------------------------------------------------
 *
 *  initialize
 *
 *      Adaptive Huffman frequency compression
 *
 *          Data structure based partly on "Application of Splay Trees
 *          to Data Compression", Communications of the ACM 8/88
 *
 *      Initialize data for compression or decompression.
 *
 *  Parameters:
 *      None.
 *
 *  Returns:
 *      Nothing
 *
 *  Notes:
 *      None.
 *
 * --------------------------------------------------------------------------
 */

void InitializeHuffman( void )
{
    register int i, j;

#if !defined(_COMPRESS_BSS_INITIALIZE)

    left = (short*)malloc( (MAXCHAR+1) * sizeof(short) );
    right = (short*)malloc( (MAXCHAR+1) * sizeof(short) );
    up = (short*)malloc( (TWICEMAX+1) * sizeof(short) );
    freq = (short*)malloc( (TWICEMAX+1) * sizeof(short) );

    copymin = (int*)malloc(COPYRANGES*sizeof(int));
    copymax = (int*)malloc(COPYRANGES*sizeof(int));

#endif

    /*
     *  Reset the Counters
     */

    input_bit_count     = 0;
    input_bit_buffer    = 0;
    output_bit_count    = 0;
    output_bit_buffer   = 0;
    bytes_in            = 0;
    bytes_out           = 0;

    /*
     *  Miscellaneous reset
     */

    insert = MINCOPY;
    dictfile = 0;
    binary = 0;

    /*
     *  Initialize Huffman frequency tree
     */

    for (i = 2; i<=TWICEMAX; i++)
    {
        up[i] = i/2;
        freq[i] = 1;
    }

    for (i = 1; i<=MAXCHAR; i++)
    {
        left[i] = 2*i;
        right[i] = 2*i+1;
    }

    /*
     *  Initialize copy distance ranges
     */

    for ( j = 0, i = 0; i < COPYRANGES; i++ )
    {
        copymin[i] = j;
        j += 1 << copybits[i];
        copymax[i] = j - 1;
    }

    maxdistance = j - 1;
    maxsize = maxdistance + MAXCOPY;

    return;
}

/* --------------------------------------------------------------------------
 *
 *  UpdateFreq
 *
 *      Update frequency counts from leaf to root
 *
 *  Parameters:
 *      None.
 *
 *  Returns:
 *      Nothing
 *
 *  Notes:
 *      None.
 *
 * --------------------------------------------------------------------------
 */

static void UpdateFreq( int a, int b )
{
    do
    {
        freq[up[a]] = freq[a] + freq[b];
        a = up[a];
        if ( a != ROOT )
        {
            if (left[up[a]] == a)
                b = right[up[a]];
            else
                b = left[up[a]];
        }
    }
    while ( a != ROOT );

    // Periodically scale frequencies down by half to avoid overflow
    // This also provides some local adaption and better compression

    if (freq[ROOT] == MAXFREQ)
    {
        for (a = 1; a<=TWICEMAX; a++)
           freq[a] >>= 1;
    }
}

/* --------------------------------------------------------------------------
 *
 *  UpdateModel
 *
 *  Description:
//      Update Huffman model for each character code
//
//  Cautions:
//      None
//
 *  Notes:
 *      None.
 *
 * --------------------------------------------------------------------------
 */

void UpdateModel( int code )
{
    register int a, b, c, ua, uua;

    a = code + SUCCMAX;
    ++freq[a];

    if (up[a] != ROOT)
    {
        ua = up[a];
        if (left[ua] == a)
            UpdateFreq(a,right[ua]);
        else
            UpdateFreq(a,left[ua]);
        do
        {
            uua = up[ua];
            if (left[uua] == ua)
                b = right[uua];
            else
            b = left[uua];

            // If high freq lower in tree, swap nodes

            if (freq[a] > freq[b])
            {
                if (left[uua] == ua)
                    right[uua] = a;
                else
                    left[uua] = a;

                if (left[ua] == a)
                {
                    left[ua] = b;
                    c = right[ua];
                }
                else
                {
                    right[ua] = b;
                    c = left[ua];
                }

                up[b] = ua;
                up[a] = uua;

                UpdateFreq(b,c);
                a = b;
            }
            a = up[a];
            ua = up[a];
        }
        while (ua != ROOT);
    }
}

//-----------------------------------------------------------------------------
//  Function:
//      AddNode
//
//  Description:
//      Add node to head of the Hash Table list
//
//  Cautions:
//      None
//
//  Functions Called:
//      None
//
//  Called By:
//
//-----------------------------------------------------------------------------

void AddNode( int n )
{
    int key;

    key = getkey(n);
    if ( head[key] == NIL )
    {
        tail[key] = n;
        succ[n] = NIL;
    }

    else
    {
        succ[n] = head[key];
        pred[head[key]] = n;
    }

    head[key] = n;
    pred[n] = NIL;
}

//-----------------------------------------------------------------------------
//  Function:
//      DeleteNode
//
//  Description:
//      Delete node from tail of list
//
//  Cautions:
//      None
//
//  Functions Called:
//      None
//
//  Called By:
//
//-----------------------------------------------------------------------------

void DeleteNode( int n )
{
    int key;

    key = getkey(n);
    if (head[key] == tail[key])
        head[key] = NIL;
    else
    {
        succ[pred[tail[key]]] = NIL;
        tail[key] = pred[tail[key]];
    }

    return;
}
