#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <time.h>
#include <windows.h>
#include "utility.h"
#include "dspos.h"

// Forward Declarations
void initialize_complex_example();
void cleanup_complex_example();
void initialize_message();
void add_errors(int num, int index);
void add_errors_to_message();
void saveDataToFile();

// Constants 
#define MAX_MSG_LENGTH 28000

// Globals
int fileSuffix = 0;

/////
// Complex Test
/////

int doComplexTest(int num_iterations)
{
   int i, j, msg_original_length_copy, error_total = 0;
   unsigned char syndrome[SYNDROME_LENGTH];

   // Setup the example's data structures
   initialize_complex_example();

   // Initilize Error-Correcting SubSystem
   initialize_ecc();

   for (i = 0; i < num_iterations; i++)
   {
      num_errors = 0;

      if (i%1000 == 0)
      {
         printf("Test #%i. Total Errors: %d\n", i, error_total);
      }

      initialize_message();

      // Reset our data pointers
      msg_original_ptr = msg_original;
      msg_encoded_ptr = msg_encoded;

      // Encode the msg buffer
      msg_original_length_copy = msg_original_length;
      while (msg_original_length_copy > 0)
      {
         encode_data(msg_original_ptr, 
                     (msg_original_length_copy >= DECODED_DATA_SIZE ? DECODED_DATA_SIZE : msg_original_length_copy), 
                     msg_encoded_ptr);
         msg_original_ptr         += DECODED_DATA_SIZE; 
         msg_original_length_copy -= DECODED_DATA_SIZE; 
         msg_encoded_ptr          += ENCODED_DATA_SIZE;
      }

      // Reset our data pointers
      msg_decoded_ptr = msg_decoded;
      msg_encoded_ptr = msg_encoded;

      // Introduce some errors
      add_errors_to_message();

      // Decode the codeword buffer
      while (msg_encoded_length > 0)
      {
         if (decode_data(msg_encoded_ptr, syndrome) != 0)
         {
	         if (correct_errors(msg_encoded_ptr, syndrome) != 0)
            {
	            /* Uh-oh! Unrecoverable bad data! */
               printf("UNRECOVERABLE ERRORS!!! Total Errors: %d\n\n", ++error_total);
               saveDataToFile();
            }
         }

         memcpy(msg_decoded_ptr, msg_encoded_ptr, DECODED_DATA_SIZE);

         msg_decoded_ptr    += DECODED_DATA_SIZE; 
         msg_encoded_length -= ENCODED_DATA_SIZE; 
         msg_encoded_ptr    += ENCODED_DATA_SIZE;
      }

      // Validate msg
      for (j = 0; j < msg_original_length; j++)
      {
         if (msg_decoded[j] != msg_original[j])
         {
            printf("DIDN'T CORRECT ERRORS!!! Total Errors: %d\n\n", ++error_total);
            saveDataToFile();
            break;
         }
      }

      Sleep(10);
   }

   cleanup_complex_example();
   printf("All Done!\nTotal Errors: %d\nTotal Bytes Encode: %I64d\nTotal Errors Inserted: %I64d\n", error_total, total_bytes_encoded, total_errors_inserted);
   return 0;
}

void initialize_complex_example()
{
   // The max msg/codeword size will be 1000 blocks, or 28000/32000 bytes
   srand((unsigned)time(NULL));
   msg_original = (unsigned char*) malloc(MAX_MSG_LENGTH);
   msg_decoded = (unsigned char*) malloc(MAX_MSG_LENGTH);
   msg_encoded = (unsigned char*) malloc(calcSizeOfEncodedBuffer(MAX_MSG_LENGTH));
   total_bytes_encoded = 0;
   total_errors_inserted = 0;
}

void cleanup_complex_example()
{
   free(msg_original);
   free(msg_decoded);
   free(msg_encoded);
}

void initialize_message()
{
   int i;

   msg_original_length = (int) ((rand() / (double) RAND_MAX) * 28000);
//   msg_original_length = 28;
   msg_encoded_length = calcSizeOfEncodedBuffer(msg_original_length);

   for (i = 0; i < msg_original_length; i++)
   {
      msg_original[i]  = 0xFF & (int) ((rand() / (double) RAND_MAX) * 256);
   }

   total_bytes_encoded += msg_original_length;
}

void add_errors(int num, int index)
{
   int i, rval;
   for (i = 0; i < num; i++)
   {
      // Documentation says rand returns 0 to RAND_MAX. It is assummed this means
      // inclusive. If we get RAND_MAX, that's a monkey wrench, since we don't want
      // rand()/RAND_MAX to return 1. 
      rval = RAND_MAX;
      while (rval == RAND_MAX)
      {
         rval = rand();
      }
      errors[num_errors].location = ((int) ((rval / (double) RAND_MAX) * 32)) + index;
      errors[num_errors].mask = (int) ((rand() / (double) RAND_MAX) * 255);
      msg_encoded[errors[num_errors].location] ^= errors[num_errors].mask;
      num_errors++;
   }
}

void add_errors_to_message()
{
   int i, num, num_segments = msg_encoded_length / 32;

   for (i = 0; i < num_segments; i++)
   {
      // Add errors according to this schedule:
      // 0 errors: 70%
      // 1 error : 20%
      // 2 errors: 10%
      num = (int) ((rand() / (double) RAND_MAX) * 100);
      if (num < 70)
      {
      } 
      else if (num < 90)
      {
         add_errors(1, i * 32);
      }
      else
      {
         add_errors(2, i * 32);
      }
   }

   total_errors_inserted += num_errors;
}

void saveDataToFile()
{
   int i;
   char filename[128];
   FILE* fp;

   sprintf(filename, "errors\\error%i.txt", fileSuffix);
   fileSuffix++;
   
   fp = fopen(filename, "w");
   if (fp == NULL)
   {
      printf("COULDN'T CREATE ERROR FILE!!!");
      return;
   }
   
   fprintf(fp, "msg_length: %i\n\n", msg_original_length);
   
   fprintf(fp, "num_errors: %i\n\n", num_errors);
   
   for (i = 0; i < num_errors; i++)
   {
      fprintf(fp, "errors[%i].location: %i\n", i, errors[i].location);
      fprintf(fp, "errors[%i].mask: 0x%02X\n\n", i, errors[i].mask);
   }
   
   fprintf(fp, "original data:\n");
   for (i = 0; i < msg_original_length; i++)
   {
      if ((i>0) && (i%26 == 0)) fprintf(fp, "\n");
      fprintf(fp, "%02X ", msg_original[i]);
   }
   fprintf(fp, "\n\n");
   
   fprintf(fp, "error data:\n");
   msg_encoded_length = calcSizeOfEncodedBuffer(msg_original_length);
   for (i = 0; i < msg_encoded_length; i++)
   {
      if ((i>0) && (i%26 == 0)) fprintf(fp, "\n");
      fprintf(fp, "%02X ", msg_encoded[i]);
   }
   
   fclose(fp);
}
