#include "reed_solomon.h"



/* Static Buffers */
byte exp_table[512];
byte log_table[256];
byte gen_poly[DOUBLE_POLY_SIZE];



void initialize_ecc()
{
   /* Initialize the galois field arithmetic tables */
   rs_init();
}

/* //;***************************************************************************
 //;---------------------------------------------------------------------------
 //;   Copyright (c) 1999 - 2002 DSP_OS Inc.  All Rights Reserved
 //;   Distribution Rights for Atmel Corporation
 //;---------------------------------------------------------------------------
 //;
 //;   Function:         encode_data
 //;
 //;   Author:           DSP OS Inc.
 //;
 //;   Date:             7/6/2002
 //;
 //;   Inputs:           uint8 * msg
 //;                     int msg_length
 //;                     uint8 * codeword
 //;
 //;   Outputs:          None
 //;
 //;   Description:      Takes in a message length up to 28 bytes. The data is
 //;                     then error encoded using reed solomon into 32 bytes.
 //;                     The error correction can correct up to 2 bad bytes.
 //;                     The 32 byte packet is placed in the user supplied buffer
 //;                     called codeword
 //;
 //;   Cautions:         None
 //;
 //;---------------------------------------------------------------------------
 //;   Revision History:
 //;
 //;   Version            Date    Modified By     Change Description
 //;   -------            ----    -----------     ------------------
 //;---------------------------------------------------------------------------
 //;*************************************************************************** */
void encode_data(unsigned char* msg, int msg_length, unsigned char* codeword)
{
   int i;
   byte c, temp;
   byte* parity = &codeword[DECODED_DATA_SIZE];

   if ((msg_length > 28)||(msg_length < 0))
   {
        return; //JUST CANT DO IT SO RETURN
   }
   /* Copy the msg into the codeword, filling empty bytes with zeros, and */
   /* clearing the parity area to zero. */
   MEMCPY(codeword, msg, msg_length);
   MEMSET(&codeword[msg_length], 0, ENCODED_DATA_SIZE - msg_length);

   /* Calculate the parity bytes */
   for (i = 0; i < DECODED_DATA_SIZE; i++)
   {
      	if ( (codeword[i] ^ parity[NUM_PARITY_BYTES-1]) != 0)
		{
         /* A fast_mult(a, b) is exp_table[log_table[a] + log_table[b]]
          * and for the next several steps, "b" is a fixed value thus as
          * an optimization we can dereference log_table[b] once and use
          * it several times below. */

            c = log_table[codeword[i] ^ parity[NUM_PARITY_BYTES-1]];
			temp = exp_table[log_table[GEN_POLY3] + c];
      	    parity[3] = parity[2]^temp;

			temp = exp_table[log_table[GEN_POLY2] + c];
      	    parity[2] = parity[1]^temp;

			temp = exp_table[log_table[GEN_POLY1] + c];
      	    parity[1] = parity[0]^temp;

      	    parity[0] = exp_table[log_table[GEN_POLY0] + c];
      	}
		else
		{
			parity[3] = parity[2];
			parity[2] = parity[1];
			parity[1] = parity[0];
			parity[0] = 0;
		}
   }
   /* REORDER THE PARITY BYTES FROM  0, 1, 2, 3 --> 3, 2, 1, 0 */
   c = parity[0];
   parity[0] = parity[3];
   parity[3] = c;
   c = parity[1];
   parity[1] = parity[2];
   parity[2] = c;
}
/*
 //;***************************************************************************
 //;---------------------------------------------------------------------------
 //;   Copyright (c) 1999 - 2002 DSP_OS Inc.  All Rights Reserved
 //;   Distribution Rights for Atmel Corporation
 //;---------------------------------------------------------------------------
 //;
 //;   Function:         decode_data
 //;
 //;   Author:           DSP OS Inc.
 //;
 //;   Date:             7/6/2002
 //;
 //;   Inputs:           uint8 * msg
 //;                     int msg_length
 //;                     uint8 * codeword
 //;
 //;   Outputs:          None
 //;
 //;   Description:      Takes in a 32 byte codeword and checks for errors. The
 //;                     syndrome table is filled out with the correct information
 //;                     so that the correct errors function can be called.
 //;
 //;   Cautions:         None
 //;
 //;---------------------------------------------------------------------------
 //;   Revision History:
 //;
 //;   Version            Date    Modified By     Change Description
 //;   -------            ----    -----------     ------------------
 //;---------------------------------------------------------------------------
 //;*************************************************************************** */


int decode_data(byte codeword[], byte syndrome[])
{
   int i, j, rc = 0;
   byte sum,temp,temp1;
   zero_poly(syndrome, SYNDROME_LENGTH);
   for (i = 0; i < NUM_PARITY_BYTES; i++)
   {
      temp1 = exp_table[i+1];
      sum = codeword[0]; /* PRELOAD TO REMOVE LOOP OF J = 0; SUM WAS CLEARED BEFORE */
      for (j = 1; j < ENCODED_DATA_SIZE; j++)
      {
		if ( sum != 0)
      	{
      		temp = exp_table[log_table[temp1] + log_table[sum]];
      		sum = codeword[j] ^ temp;
      	}
      	else
      	{
      		/* temp = 0; */
      		sum = codeword[j]; //ANYTHING ^ WITH 0 IS ITSELF
      	}
      	/* sum = codeword[j] ^ temp; */
      }

      rc += (syndrome[i] = sum);
   }
   return rc;
}
/*
 //;***************************************************************************
 //;---------------------------------------------------------------------------
 //;   Copyright (c) 1999 - 2002 DSP_OS Inc.  All Rights Reserved
 //;   Distribution Rights for Atmel Corporation
 //;---------------------------------------------------------------------------
 //;
 //;   Function:         correct_errors
 //;
 //;   Author:           DSP OS Inc.
 //;
 //;   Date:             7/6/2002
 //;
 //;   Inputs:           uint8 * msg
 //;                     int msg_length
 //;                     uint8 * codeword
 //;
 //;   Outputs:          int
 //;
 //;   Description:      Takes in a 32 byte codeword and checks for errors. The
 //;                     syndrome table is filled out with the correct information
 //;                     so that the correct errors function can be called.
 //;
 //;   Cautions:         None
 //;
 //;---------------------------------------------------------------------------
 //;   Revision History:
 //;
 //;   Version            Date    Modified By     Change Description
 //;   -------            ----    -----------     ------------------
 //;---------------------------------------------------------------------------
 //;*************************************************************************** */

int correct_errors(byte codeword[], byte syndrome[])
{
   int r, i, j, err, num_errors;
   byte num, denom;
   byte error_locations[NUM_PARITY_BYTES];
   byte lambda[POLY_SIZE], omega[POLY_SIZE];

   Modified_Berlekamp_Massey(syndrome, lambda, omega);
   num_errors = Find_Roots(lambda, error_locations);

   if (num_errors > MAX_CORRECTABLE_ERRORS)
   {
		#ifdef RS_DEBUG
	   	 	fprintf(stderr, "Uncorrectable codeword!\n");
	   	#endif
      return 1;
   }

   for (r = 0; r < num_errors; r++)
   {
      /* first check for illegal error locs */
      if (error_locations[r] >= ENCODED_DATA_SIZE)  /* Modified from DECODED_DATA_SIZE */
      {
         return 1;
      }

      i = error_locations[r];

      /* evaluate Omega at alpha^(-i) */
      num = 0;
      for (j = 0; j < SYNDROME_LENGTH; j++)
      {
         num ^= fast_mult(omega[j], exp_table[((255 - i) * j) % 255]);
      }

      /* evaluate Lambda' (derivative) at alpha^(-i) ; all odd powers disappear */
      denom = 0;
      for (j = 1; j < SYNDROME_LENGTH; j += 2)
      {
         denom ^= fast_mult(lambda[j], exp_table[((255 - i ) * (j - 1)) & 0xFF]);
      }

      err = fast_mult(num, fast_inv(denom));

      codeword[ENCODED_DATA_SIZE - i - 1] ^= err;
   }

   return 0;
}

void rs_init(void)
{
   int i, z;
   int pinit,p1,p2,p3,p4,p5,p6,p7,p8;
   byte tp[POLY_SIZE], tp1[POLY_SIZE];

   /* Initialize exponent and logarithm tables */

   /* This is one of 14 irreducible polynomials
    * of degree 8 and cycle length 255. (Ch 5, pp. 275, Magnetic Recording)
    * The high order 1 bit is implicit */

   /* x^8 + x^4 + x^3 + x^2 + 1 */

   pinit = p2 = p3 = p4 = p5 = p6 = p7 = p8 = 0;
   p1 = 1;

   exp_table[0] = 1;
   exp_table[255] = exp_table[0];
   log_table[0] = 0;

   for (i = 1; i < 256; i++)
   {
      pinit = p8;
      p8 = p7;
      p7 = p6;
      p6 = p5;
      p5 = p4 ^ pinit;
      p4 = p3 ^ pinit;
      p3 = p2 ^ pinit;
      p2 = p1;
      p1 = pinit;
      exp_table[i] = p1 + p2*2 + p3*4 + p4*8 + p5*16 + p6*32 + p7*64 + p8*128;
      exp_table[i+255] = exp_table[i];
   }

   for (i = 1; i < 256; i++)
   {
      for (z = 0; z < 256; z++)
      {
         if (exp_table[z] == i)
         {
            log_table[i] = z;
            break;
         }
      }
   }

   /* Create a generator polynomial for an n byte RS code.
    * The coefficients are returned in the genPoly arg.
    * Make sure that the genPoly array which is passed in is
    * at least n+1 bytes long.
    *
    * multiply (x + a^n) for n = 1 to nbytes
    */

   zero_poly(tp1, POLY_SIZE);
   tp1[0] = 1;

   for (i = 1; i <= NUM_PARITY_BYTES; i++)
   {
      zero_poly(tp, POLY_SIZE);
      tp[0] = exp_table[i];	/* set up x+a^n */
      tp[1] = 1;

      mult_polys(gen_poly, tp, tp1);
      copy_poly(tp1, gen_poly, POLY_SIZE);
   }
}

void Modified_Berlekamp_Massey(byte syndrome[], byte lambda[], byte omega[])
{
   int i, n, L = 0, L2, k = -1;
   byte d, psi[POLY_SIZE], psi2[POLY_SIZE], D[POLY_SIZE];

   zero_poly(D, POLY_SIZE);
   D[0] = 1;
   mul_z_poly(D);

   zero_poly(psi, POLY_SIZE);
   psi[0] = 1;

   for (n = 0; n < NUM_PARITY_BYTES; n++)
   {
      d = compute_discrepancy(psi, syndrome, L, n);

      if (d != 0)
      {
         /* psi2 = psi - d*D */
         for (i = 0; i < POLY_SIZE; i++)
         {
            psi2[i] = psi[i] ^ fast_mult(d, D[i]);
         }

         if (L < (n-k))
         {
            L2 = n-k;
            k = n;
            for (i = 0; i < POLY_SIZE; i++)
            {
               D[i] = fast_mult(psi[i], fast_inv(d));
            }
            L = L2;
         }

         copy_poly(psi, psi2, POLY_SIZE);
      }

      mul_z_poly(D);
   }

   copy_poly(lambda, psi, POLY_SIZE);

   compute_modified_omega(syndrome, lambda, omega);
}

/* multiply by z, i.e., shift right by 1 */
void mul_z_poly(byte src[])
{
  int i;
  for (i = POLY_SIZE - 1; i > 0; i--)
  {
     src[i] = src[i-1];
  }
  src[0] = 0;
}

byte compute_discrepancy(byte lambda[], byte syndrome[], int L, int n)
{
  int i;
  byte sum=0;

  for (i = 0; i <= L; i++)
  {
    sum ^= fast_mult(lambda[i], syndrome[n-i]);
  }

  return sum;
}

/*
 given Psi (called Lambda in Modified_Berlekamp_Massey) and synBytes,
 compute the combined erasure/error evaluator polynomial as Psi*S mod z^4
*/
void compute_modified_omega(byte syndrome[], byte lambda[], byte omega[])
{
   byte product[DOUBLE_POLY_SIZE];

   mult_polys(product, lambda, syndrome);
   zero_poly(omega, POLY_SIZE);
   copy_poly(omega, product, NUM_PARITY_BYTES);
}

int Find_Roots(byte lambda[], byte error_locations[])
{
   int sum, i, j, errors = 0;

   for (i = 1; i < 255; i++)
   {
      sum = 0;

      for (j = 0; j < NUM_PARITY_BYTES + 1; j++)
      {
         sum ^= fast_mult(exp_table[(j * i) % 255], lambda[j]);
      }

      if (sum == 0)
      {
         error_locations[errors] = (255-i);
         errors++;

         /* Don't bother finding more errors then we can correct */
         if (errors > NUM_PARITY_BYTES)
         {
            break;
         }
      }
   }

   return errors;
}

/********** polynomial arithmetic *******************/


void mult_polys(byte dst[], byte p1[], byte p2[])
{
   int i, j;
   byte tmp1[DOUBLE_POLY_SIZE];

   zero_poly(dst, DOUBLE_POLY_SIZE);

   for (i = 0; i < POLY_SIZE; i++)
   {
      zero_poly(&tmp1[POLY_SIZE], POLY_SIZE);

      /* scale tmp1 by p1[i] */
      for(j = 0; j < POLY_SIZE; j++)
      {
         tmp1[j] = fast_mult(p2[j], p1[i]);
      }

      /* and mult (shift) tmp1 right by i */
      for (j = DOUBLE_POLY_SIZE - 1; j >= i; j--)
      {
         tmp1[j] = tmp1[j-i];
      }

      zero_poly(tmp1, i);

      /* add into partial product */
      for (j = 0; j < DOUBLE_POLY_SIZE; j++)
      {
         dst[j] ^= tmp1[j];
      }
   }
}
/*
 //;***************************************************************************
 //;---------------------------------------------------------------------------
 //;   Copyright (c) 1999 - 2002 DSP_OS Inc.  All Rights Reserved
 //;   Distribution Rights for Atmel Corporation
 //;---------------------------------------------------------------------------
 //;
 //;   Function:         rs_memcpy and rs_memset
 //;
 //;   Author:           DSP OS Inc.
 //;
 //;   Date:             7/6/2002
 //;
 //;   Inputs:           uint8 * destination
 //;                     uint8 * source
 //;                     int    size
 //;
 //;   Outputs:          None
 //;
 //;   Description:      These functions perform a mem copy and memset
 //;
 //;   Cautions:         None
 //;
 //;---------------------------------------------------------------------------
 //;   Revision History:
 //;
 //;   Version            Date    Modified By     Change Description
 //;   -------            ----    -----------     ------------------
 //;---------------------------------------------------------------------------
 //;*************************************************************************** */
void  rs_memcpy(unsigned char * destination,unsigned char * source, int size)
{
	int i;
	if (size == 0)
    {
        return;
    }
    for (i = 0; i < size; i++)
    {
        *(destination)++ = *(source)++;

    }
    return;
}

void  rs_memset(unsigned char * destination,unsigned char value, int size)
{
	int i;
	if (size == 0)
    {
        return;
    }
    for (i = 0; i < size; i++)
    {
        *(destination)++ = value;

    }
    return;
}
